"use strict";
const requestLib = require("request").defaults({ jar: true, proxy: process.env.FB_PROXY });
const log = require("npmlog");
const stream = require('stream');

const request = promisify(requestLib);

// ========== Utils ==========
const delay = ms => new Promise(r => setTimeout(r, ms));
const getType = obj => Object.prototype.toString.call(obj).slice(8, -1);

function getGUID() {
    const section = () =>
        Math.floor((1 + Math.random()) * 0x10000)
            .toString(16)
            .substring(1);
    return (
        section() +
        section() +
        "-" +
        section() +
        "-" +
        section() +
        "-" +
        section() +
        "-" +
        section() +
        section() +
        section()
    );
}

const formatID = id => id?.replace(/(fb)?id[:.]/, "") ?? id;
const formatCookie = ([k, v, , p], d) => `${k}=${v}; Path=${p}; Domain=${d}.com`;

const getFrom = (str, s, e) => {
    const i = str.indexOf(s); if (i < 0) return "";
    const j = str.indexOf(e, i + s.length);
    if (j < 0) throw new Error("End token not found");
    return str.substring(i + s.length, j);
};

function makeParsable(html) {
    const withoutForLoop = html.replace(/for\s*\(\s*;\s*;\s*\)\s*;\s*/, "");
    const maybeMultipleObjects = withoutForLoop.split(/\}\r\n *\{/);
    if (maybeMultipleObjects.length === 1) return maybeMultipleObjects;
    return "[" + maybeMultipleObjects.join("},{") + "]";
}

// ========== Promisify ==========
function promisify(fn) {
    const out = typeof fn === "function" && /callback|cb\)/.test(fn.toString().split("\n")[0]) ? (...a) => new Promise((res, rej) => fn(...a, (e, d) => e ? rej(e) : res(d))) : fn;
    for (const k of Object.keys(fn))
        out[k] = typeof fn[k] === "function" && /callback|cb\)/.test(fn[k].toString()) ? (...a) => new Promise((res, rej) => fn[k](...a, (e, d) => e ? rej(e) : res(d))) : fn[k];
    return out;
}

// ========== HTTP helpers ==========
const getHeaders = (url, opt, ctx, custom) => ({
    "Content-Type": "application/x-www-form-urlencoded",
    Referer: "https://www.facebook.com/",
    Host: new URL(url).host,
    Origin: "https://www.facebook.com",
    "User-Agent": opt?.userAgent || "Mozilla/5.0",
    Connection: "keep-alive",
    "sec-fetch-site": "same-origin",
    ...(ctx?.region && { "X-MSGR-Region": ctx.region }),
    ...custom
});

const get = (url, jar, qs, opt, ctx) => request({ url, jar, qs, headers: getHeaders(url, opt, ctx), timeout: 6e4, method: "GET", gzip: true }).then(r => Array.isArray(r) ? r[0] : r);

const post = (url, jar, form, opt, ctx, custom) => request({ url, jar, form, headers: getHeaders(url, opt, ctx, custom), timeout: 6e4, method: "POST", gzip: true }).then(r => Array.isArray(r) ? r[0] : r);

const postFormData = (url, jar, form, qs, opt, ctx) =>
    request({
        url, jar, formData: form, qs,
        headers: { ...getHeaders(url, opt, ctx), "Content-Type": "multipart/form-data" },
        timeout: 6e4, method: "POST", gzip: true
    }).then(r => Array.isArray(r) ? r[0] : r);

const saveCookies = jar => res => {
    (res.headers["set-cookie"] || []).forEach(c => {
        if (c.includes(".facebook.com")) {
            jar.setCookie(c, "https://www.facebook.com");
            jar.setCookie(c.replace(/domain=\.facebook\.com/, "domain=.messenger.com"), "https://www.messenger.com");
        }
    });
    return res;
};

const getAppState = jar => jar.getCookies("https://www.facebook.com").concat(jar.getCookies("https://www.messenger.com"));

// ========== Facebook-specific ==========
const makeDefaults = (html, userID, ctx) => {
    const fb_dtsg = getFrom(html, '"DTSGInitData",[],{"token":"', '",');
    const ttstamp = "2" + [...fb_dtsg].map(c => c.charCodeAt(0)).join('');
    const revision = getFrom(html, 'revision":', ",");
    let req = 1;
    const add = o => ({ av: userID, __user: userID, __req: (req++).toString(36), __rev: revision, __a: 1, fb_dtsg, jazoest: ttstamp, ...o });
    return {
        get: (url, jar, qs, cx) => get(url, jar, add(qs), ctx.globalOptions, cx || ctx),
        post: (url, jar, form, cx) => post(url, jar, add(form), ctx.globalOptions, cx || ctx),
        postFormData: (url, jar, form, qs, cx) => postFormData(url, jar, add(form), add(qs), ctx.globalOptions, cx || ctx)
    };
};
function tryPromise(tryFunc) {
    return new Promise((resolve, reject) => {
        try {
            resolve(tryFunc());
        } catch (error) {
            reject(error);
        }
    });
}

function parseAndCheckLogin(ctx, def, retry = 0, src = (() => { try { throw new Error() } catch (e) { return e } })()) {
    return data => tryPromise(() => {
        log.verbose("parseAndCheckLogin", data.body);
        if (data.statusCode >= 500 && retry < 5) {
            const t = Math.random() * 5000 | 0, url = `${data.request.uri.protocol}//${data.request.uri.hostname}${data.request.uri.pathname}`;
            const post = data.request.headers["Content-Type"].startsWith("multipart") ? def.postFormData : def.post;
            log.warn("parseAndCheckLogin", `Status ${data.statusCode} retry ${retry + 1} in ${t}ms`);
            return delay(t).then(() => post(url, ctx.jar, data.request.formData)).then(parseAndCheckLogin(ctx, def, retry + 1, src));
        }
        if (data.statusCode !== 200) throw new CustomError({ message: `Status ${data.statusCode}`, statusCode: data.statusCode, res: data.body, sourceCall: src });
        let res; try { res = JSON.parse(makeParsable(data.body)) } catch (e) { throw new CustomError({ message: "JSON.parse error", detail: e, res: data.body, sourceCall: src }) }

        if (res.jsmods?.require) for (const i of res.jsmods.require)
            if (i[0] === "ServerRedirect" && i[1] === "redirectPageTo") { const m = i[3]?.[0]?.match(/\/checkpoint\/(\d+)\//); if (m) { res.die = true; res.type = m[1]; break } }

        if (res.error === 1357001) res.logout = true;
        if (res.redirect && data.request.method === "GET") return def.get(res.redirect, ctx.jar).then(parseAndCheckLogin(ctx, def, undefined, src));

        if (res.jsmods?.require?.[0]?.[0] === "Cookie") {
            ctx.jar.setCookie(formatCookie(res.jsmods.require[0][3], "facebook"), "https://www.facebook.com");
            ctx.jar.setCookie(formatCookie(res.jsmods.require[0][3], "messenger"), "https://www.messenger.com");
        }
        for (const i of res.jsmods?.require || [])
            if (i[0] === "DTSG" && i[1] === "setToken") { ctx.fb_dtsg = i[3][0]; ctx.ttstamp = "2" + [...ctx.fb_dtsg].map(c => c.charCodeAt()).join("") }

        return res;
    });
}

function formatAttachment(attachment1, attachment2) {
    attachment2 = attachment2 || { id: "", image_data: {} };
    attachment1 = attachment1.mercury ? attachment1.mercury : attachment1;
    var blob = attachment1.blob_attachment;
    var type =
        blob && blob.__typename ? blob.__typename : attachment1.attach_type;
    if (!type && attachment1.sticker_attachment) {
        type = "StickerAttachment";
        blob = attachment1.sticker_attachment;
    } else if (!type && attachment1.extensible_attachment) {
        if (
            attachment1.extensible_attachment.story_attachment &&
            attachment1.extensible_attachment.story_attachment.target &&
            attachment1.extensible_attachment.story_attachment.target.__typename &&
            attachment1.extensible_attachment.story_attachment.target.__typename === "MessageLocation"
        ) type = "MessageLocation";
        else type = "ExtensibleAttachment";

        blob = attachment1.extensible_attachment;
    }
    switch (type) {
        case "sticker":
            return {
                type: "sticker",
                ID: attachment1.metadata.stickerID.toString(),
                url: attachment1.url,

                packID: attachment1.metadata.packID.toString(),
                spriteUrl: attachment1.metadata.spriteURI,
                spriteUrl2x: attachment1.metadata.spriteURI2x,
                width: attachment1.metadata.width,
                height: attachment1.metadata.height,

                caption: attachment2.caption,
                description: attachment2.description,

                frameCount: attachment1.metadata.frameCount,
                frameRate: attachment1.metadata.frameRate,
                framesPerRow: attachment1.metadata.framesPerRow,
                framesPerCol: attachment1.metadata.framesPerCol,

                stickerID: attachment1.metadata.stickerID.toString(), // @Legacy
                spriteURI: attachment1.metadata.spriteURI, // @Legacy
                spriteURI2x: attachment1.metadata.spriteURI2x // @Legacy
            };
        case "file":
            return {
                type: "file",
                filename: attachment1.name,
                ID: attachment2.id.toString(),
                url: attachment1.url,

                isMalicious: attachment2.is_malicious,
                contentType: attachment2.mime_type,

                name: attachment1.name, // @Legacy
                mimeType: attachment2.mime_type, // @Legacy
                fileSize: attachment2.file_size // @Legacy
            };
        case "photo":
            return {
                type: "photo",
                ID: attachment1.metadata.fbid.toString(),
                filename: attachment1.fileName,
                thumbnailUrl: attachment1.thumbnail_url,

                previewUrl: attachment1.preview_url,
                previewWidth: attachment1.preview_width,
                previewHeight: attachment1.preview_height,

                largePreviewUrl: attachment1.large_preview_url,
                largePreviewWidth: attachment1.large_preview_width,
                largePreviewHeight: attachment1.large_preview_height,

                url: attachment1.metadata.url, // @Legacy
                width: attachment1.metadata.dimensions.split(",")[0], // @Legacy
                height: attachment1.metadata.dimensions.split(",")[1], // @Legacy
                name: attachment1.fileName // @Legacy
            };
        case "animated_image":
            return {
                type: "animated_image",
                ID: attachment2.id.toString(),
                filename: attachment2.filename,

                previewUrl: attachment1.preview_url,
                previewWidth: attachment1.preview_width,
                previewHeight: attachment1.preview_height,

                url: attachment2.image_data.url,
                width: attachment2.image_data.width,
                height: attachment2.image_data.height,

                name: attachment1.name, // @Legacy
                facebookUrl: attachment1.url, // @Legacy
                thumbnailUrl: attachment1.thumbnail_url, // @Legacy
                mimeType: attachment2.mime_type, // @Legacy
                rawGifImage: attachment2.image_data.raw_gif_image, // @Legacy
                rawWebpImage: attachment2.image_data.raw_webp_image, // @Legacy
                animatedGifUrl: attachment2.image_data.animated_gif_url, // @Legacy
                animatedGifPreviewUrl: attachment2.image_data.animated_gif_preview_url, // @Legacy
                animatedWebpUrl: attachment2.image_data.animated_webp_url, // @Legacy
                animatedWebpPreviewUrl: attachment2.image_data.animated_webp_preview_url // @Legacy
            };
        case "share":
            return {
                type: "share",
                ID: attachment1.share.share_id.toString(),
                url: attachment2.href,

                title: attachment1.share.title,
                description: attachment1.share.description,
                source: attachment1.share.source,

                image: attachment1.share.media.image,
                width: attachment1.share.media.image_size.width,
                height: attachment1.share.media.image_size.height,
                playable: attachment1.share.media.playable,
                duration: attachment1.share.media.duration,

                subattachments: attachment1.share.subattachments,
                properties: {},

                animatedImageSize: attachment1.share.media.animated_image_size, // @Legacy
                facebookUrl: attachment1.share.uri, // @Legacy
                target: attachment1.share.target, // @Legacy
                styleList: attachment1.share.style_list // @Legacy
            };
        case "video":
            return {
                type: "video",
                ID: attachment1.metadata.fbid.toString(),
                filename: attachment1.name,

                previewUrl: attachment1.preview_url,
                previewWidth: attachment1.preview_width,
                previewHeight: attachment1.preview_height,

                url: attachment1.url,
                width: attachment1.metadata.dimensions.width,
                height: attachment1.metadata.dimensions.height,

                duration: attachment1.metadata.duration,
                videoType: "unknown",

                thumbnailUrl: attachment1.thumbnail_url // @Legacy
            };
        case "error":
            return {
                type: "error",

                // Save error attachments because we're unsure of their format,
                // and whether there are cases they contain something useful for debugging.
                attachment1: attachment1,
                attachment2: attachment2
            };
        case "MessageImage":
            return {
                type: "photo",
                ID: blob.legacy_attachment_id,
                filename: blob.filename,
                thumbnailUrl: blob.thumbnail.uri,

                previewUrl: blob.preview.uri,
                previewWidth: blob.preview.width,
                previewHeight: blob.preview.height,

                largePreviewUrl: blob.large_preview.uri,
                largePreviewWidth: blob.large_preview.width,
                largePreviewHeight: blob.large_preview.height,

                url: blob.large_preview.uri, // @Legacy
                width: blob.original_dimensions.x, // @Legacy
                height: blob.original_dimensions.y, // @Legacy
                name: blob.filename // @Legacy
            };
        case "MessageAnimatedImage":
            return {
                type: "animated_image",
                ID: blob.legacy_attachment_id,
                filename: blob.filename,

                previewUrl: blob.preview_image.uri,
                previewWidth: blob.preview_image.width,
                previewHeight: blob.preview_image.height,

                url: blob.animated_image.uri,
                width: blob.animated_image.width,
                height: blob.animated_image.height,

                thumbnailUrl: blob.preview_image.uri, // @Legacy
                name: blob.filename, // @Legacy
                facebookUrl: blob.animated_image.uri, // @Legacy
                rawGifImage: blob.animated_image.uri, // @Legacy
                animatedGifUrl: blob.animated_image.uri, // @Legacy
                animatedGifPreviewUrl: blob.preview_image.uri, // @Legacy
                animatedWebpUrl: blob.animated_image.uri, // @Legacy
                animatedWebpPreviewUrl: blob.preview_image.uri // @Legacy
            };
        case "MessageVideo":
            return {
                type: "video",
                filename: blob.filename,
                ID: blob.legacy_attachment_id,

                previewUrl: blob.large_image.uri,
                previewWidth: blob.large_image.width,
                previewHeight: blob.large_image.height,

                url: blob.playable_url,
                width: blob.original_dimensions.x,
                height: blob.original_dimensions.y,

                duration: blob.playable_duration_in_ms,
                videoType: blob.video_type.toLowerCase(),

                thumbnailUrl: blob.large_image.uri // @Legacy
            };
        case "MessageAudio":
            return {
                type: "audio",
                filename: blob.filename,
                ID: blob.url_shimhash,

                audioType: blob.audio_type,
                duration: blob.playable_duration_in_ms,
                url: blob.playable_url,

                isVoiceMail: blob.is_voicemail
            };
        case "StickerAttachment":
            return {
                type: "sticker",
                ID: blob.id,
                url: blob.url,

                packID: blob.pack ? blob.pack.id : null,
                spriteUrl: blob.sprite_image,
                spriteUrl2x: blob.sprite_image_2x,
                width: blob.width,
                height: blob.height,

                caption: blob.label,
                description: blob.label,

                frameCount: blob.frame_count,
                frameRate: blob.frame_rate,
                framesPerRow: blob.frames_per_row,
                framesPerCol: blob.frames_per_column,

                stickerID: blob.id, // @Legacy
                spriteURI: blob.sprite_image, // @Legacy
                spriteURI2x: blob.sprite_image_2x // @Legacy
            };
        case "MessageLocation":
            var urlAttach = blob.story_attachment.url;
            var mediaAttach = blob.story_attachment.media;

            var u = querystring.parse(url.parse(urlAttach).query).u;
            var where1 = querystring.parse(url.parse(u).query).where1;
            var address = where1.split(", ");

            var latitude;
            var longitude;

            try {
                latitude = Number.parseFloat(address[0]);
                longitude = Number.parseFloat(address[1]);
            } catch (err) {
                /* empty */

            }

            var imageUrl;
            var width;
            var height;

            if (mediaAttach && mediaAttach.image) {
                imageUrl = mediaAttach.image.uri;
                width = mediaAttach.image.width;
                height = mediaAttach.image.height;
            }

            return {
                type: "location",
                ID: blob.legacy_attachment_id,
                latitude: latitude,
                longitude: longitude,
                image: imageUrl,
                width: width,
                height: height,
                url: u || urlAttach,
                address: where1,

                facebookUrl: blob.story_attachment.url, // @Legacy
                target: blob.story_attachment.target, // @Legacy
                styleList: blob.story_attachment.style_list // @Legacy
            };
        case "ExtensibleAttachment":
            return {
                type: "share",
                ID: blob.legacy_attachment_id,
                url: blob.story_attachment.url,

                title: blob.story_attachment.title_with_entities.text,
                description: blob.story_attachment.description &&
                    blob.story_attachment.description.text,
                source: blob.story_attachment.source ? blob.story_attachment.source.text : null,

                image: blob.story_attachment.media &&
                    blob.story_attachment.media.image &&
                    blob.story_attachment.media.image.uri,
                width: blob.story_attachment.media &&
                    blob.story_attachment.media.image &&
                    blob.story_attachment.media.image.width,
                height: blob.story_attachment.media &&
                    blob.story_attachment.media.image &&
                    blob.story_attachment.media.image.height,
                playable: blob.story_attachment.media &&
                    blob.story_attachment.media.is_playable,
                duration: blob.story_attachment.media &&
                    blob.story_attachment.media.playable_duration_in_ms,
                playableUrl: blob.story_attachment.media == null ? null : blob.story_attachment.media.playable_url,

                subattachments: blob.story_attachment.subattachments,
                properties: blob.story_attachment.properties.reduce(function (/** @type {{ [x: string]: any; }} */obj, /** @type {{ key: string | number; value: { text: any; }; }} */cur) {
                    obj[cur.key] = cur.value.text;
                    return obj;
                }, {}),

                facebookUrl: blob.story_attachment.url, // @Legacy
                target: blob.story_attachment.target, // @Legacy
                styleList: blob.story_attachment.style_list // @Legacy
            };
        case "MessageFile":
            return {
                type: "file",
                filename: blob.filename,
                ID: blob.message_file_fbid,

                url: blob.url,
                isMalicious: blob.is_malicious,
                contentType: blob.content_type,

                name: blob.filename,
                mimeType: "",
                fileSize: -1
            };
        default:
            throw new Error(
                "unrecognized attach_file of type " +
                type +
                "`" +
                JSON.stringify(attachment1, null, 4) +
                " attachment2: " +
                JSON.stringify(attachment2, null, 4) +
                "`"
            );
    }
}

// ========== CustomError ==========
class CustomError extends Error {
    constructor(obj) {
        if (typeof obj === 'string')
            obj = { message: obj };
        if (typeof obj !== 'object' || obj === null)
            throw new TypeError('Object required');
        obj.message ? super(obj.message) : super();
        Object.assign(this, obj);
    }
}

function markDelivery(ctx, api, threadID, messageID) {
    if (threadID && messageID) {
        api.markAsDelivered(threadID, messageID, (err) => {
            if (err) log.error("markAsDelivered", err);
            else if (ctx.globalOptions.autoMarkRead) {
                api.markAsRead(threadID, (err2) => {
                    if (err2) log.error("markAsRead", err2);
                });
            }
        });
    }
}

function getAdminTextMessageType(m) {
    switch (m.type) {
        case "joinable_group_link_mode_change":
            return "log:link-status";
        case "magic_words":
            return "log:magic-words";
        case "change_thread_theme":
            return "log:thread-color";
        case "change_thread_quick_reaction":
            return "log:thread-icon";
        case "change_thread_nickname":
            return "log:user-nickname";
        case "change_thread_admins":
            return "log:thread-admins";
        case "group_poll":
            return "log:thread-poll";
        case "change_thread_approval_mode":
            return "log:thread-approval-mode";
        case "messenger_call_log":
        case "participant_joined_group_call":
            return "log:thread-call";
        case "pin_messages_v2":
            return "log:thread-pinned";
        case "joinable_group_link_reset":
            return "joinable_group_link_reset"

    }
}

function formatDeltaEvent(m) {
    var logMessageType;
    var logMessageData;
    var text_new;
    switch (m.class) {
        case "JoinableMode":
            if (m.mode) return;
            logMessageType = "joinable_group_link_reset"
            logMessageData = { link: m.link }
            break;
        case "AdminTextMessage":
            logMessageType = getAdminTextMessageType(m);
            logMessageData = m.untypedData;
            break;
        case "ThreadName":
            logMessageType = "log:thread-name";
            logMessageData = { name: m.name };
            break;
        case "ParticipantsAddedToGroupThread":
            logMessageType = "log:subscribe";
            logMessageData = { addedParticipants: m.addedParticipants };
            break;
        case "ParticipantLeftGroupThread":
            logMessageType = "log:unsubscribe";
            logMessageData = { leftParticipantFbId: m.leftParticipantFbId };
            break;

    }

    return {
        type: "event",
        threadID: formatID((m.messageMetadata.threadKey.threadFbId || m.messageMetadata.threadKey.otherUserFbId).toString()),
        logMessageType: logMessageType,
        logMessageData: logMessageData,
        logMessageBody: m.messageMetadata.adminText || text_new || null,
        author: m.messageMetadata.actorFbId,
        participantIDs: (m.participants || []).map(p => p.toString())
    };
}

function isReadableStream(obj) { return obj instanceof stream.Stream; }

function binaryToDecimal(data) {
    let ret = "";
    while (data !== "0") {
        let end = 0;
        let fullName = "";
        let i = 0;
        for (; i < data.length; i++) {
            end = 2 * end + parseInt(data[i], 10);
            if (end >= 10) {
                fullName += "1";
                end -= 10;
            }
            else {
                fullName += "0";
            }
        }
        ret = end.toString() + ret;
        data = fullName.slice(fullName.indexOf("1"));
    }
    return ret;
}

function generateOfflineThreadingID() {
    const ret = Date.now();
    const value = Math.floor(Math.random() * 4294967295);
    const str = ("0000000000000000000000" + value.toString(2)).slice(-22);
    const msgs = ret.toString(2) + str;
    return binaryToDecimal(msgs);
}

function padZeros(val, len) {
    val = String(val);
    len = len || 2;
    while (val.length < len) val = "0" + val;
    return val;
}

function generateTimestampRelative() {
    const d = new Date();
    return d.getHours() + ":" + padZeros(d.getMinutes());
}

function getSignatureID() {
    return Math.floor(Math.random() * 2147483648).toString(16);
}

function generateThreadingID(clientID) {
    const k = Date.now();
    const l = Math.floor(Math.random() * 4294967295);
    const m = clientID;
    return "<" + k + ":" + l + "-" + m + "@mail.projektitan.com>";
}

// ========== Exports ==========
module.exports = {
    getJar: request.jar,
    saveCookies,
    makeDefaults,
    getAppState,
    parseAndCheckLogin,
    get,
    getType,
    getHeaders,
    getGUID,
    formatID,
    postFormData,
    formatAttachment,
    markDelivery,
    formatDeltaEvent,
    isReadableStream,
    generateOfflineThreadingID,
    generateTimestampRelative,
    getSignatureID,
    generateThreadingID,
};