% ---------------------------------------
% (RANDOM) DOTS SEQUENCES
%
% See additional demos in BasicVideoTools:
%   demo_motion_programs.m 
%   example_rendom_dots_sequence.m
% See additional motion estimation toolbox:
%   http://isp.uv.es/Video_coding.html
% ----------------------------------------
%
% The basic procedure to generate a random-dots sequence is the following: 
%
% (0) Initialization: 
%     Take a number of objects (rectangles of certain size and color) at 
%     random locations.
%
% (1) Generate an image (frame of the sequence) from the objects data
%     (size, color and location).
%
% (2) Compute (or assign) the 2d (retinal) speed of each object.
%
% (3) Update the locations at a future frame according to the (known) speed 
%     of each object: x(t+1) = x(t) + v(x,t)/ft
%     where x(t) and v(x,t) are the location and speed at time t, and ft is
%     the temporal sampling frequency.
%     We assume that size and color of objects are preserved along the sequence. 
%     Go to step (1) and repeat.
%
% BasicVideoTools comes with functions to implement the non-trivial steps:
%
% (1) generate_frame.m
%
% (2) Characteristic spatially-variant flow fields where v can be obtained from x:
%
%       radial_flow.m  (as in ego-motion parallel to the optical axis)
%      lateral_flow.m  (as in ego-motion normal to the optical axis)     
%     circular_flow.m  (the headache-psychic flow)
%   sinusoidal_flow.m  (for those that love to analyze everything into Fourier components)
%      
%     For physics-based fields see XXXX
%
% And a convenient function to call these in a loop:  dots_sequence.m
%
% Here you have an example on how to use them to generate sequences.
%


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%
%% INITIALIZATION:
%%   Sequence parameters (sampling frequencies and spatio-temporal size)
%%   Object parameters   (sizes, colors and initial spatial locations)
%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

   % SEQUENCE PARAMETERS    
   % Spatial size (in pixels). We assume squared frames.
   % Temporal size (in number of frames).
   % Sampling frequencies (spatial in cycles per degree, and temporal in Hz)
    
        Nx = 75;    % pixels
        Nt = 64;    % frames
        fs = Nx/2;  % This choice implies that the size of the frames is 2 deg.
        ft = 20;

   % PARAMETERS OF THE OBJECTS
   % Number of objects, sizes, colors (only luminance level) and locations.
   %
   % In order to choose these parameters (mainly number and location) you
   % have to consider that the selected flow field (e.g. radial expansive flow) 
   % will be applied to the objects, and hence they may be expelled from
   % the image after some frames. 
   % Similarly, if you plan to apply a translation flow, some objects have
   % to be placed outside the field of view ready to jump in after the 
   % displacements that will happen in some delta_t = 1/ft
   %
    
   % NUMBER
     N_objects = 4000;   % Number of objects

   % SIZES in degrees. Note that the size of the frame in degrees is Nx/fs
   %                   Subpixel sizes (small dots) are fine. Their energy
   %                   will spread out over the pixel with the
   %                   corresponding reduction in amplitude (see details in generate_image.m).
   %

     % Range of sizes (will be random)
     S_max=0.04; % in deg
     S_min=0.03; % in deg

     widths = S_min + (S_max-S_min)*rand(N_objects,1);
    heights = S_min + (S_max-S_min)*rand(N_objects,1);
      sizes = [widths heights];

   % COLOR (gray level)

      Colormax=255;
      Colormin=70;
      colors = Colormin + ceil((Colormax-Colormin)*rand(N_objects,1));

   %  LOCATIONS in degrees. (random in this example):
   %                        Note that the origin of the spatial domain is
   %                        at the top-left corner, and the center of the
   %                        frame is at [Nx/fs Nx/fs]/2

      locations_centered = [(Nx/fs)*rand(N_objects,2) - repmat([Nx/(2*fs) Nx/(2*fs)],N_objects,1)]; % These cover the spatial extension but contain negative numbers
        locations_scaled = 4*locations_centered;                                                    % Apply whatever scale factor you like (bigger if speed is big) 
       initial_locations = locations_scaled + repmat([Nx/(2*fs) Nx/(2*fs)],N_objects,1);            % Re-center

               
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%
%% SEQUENCE GENERATION:
%%   Initial data + Flow data
%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

   % Build your flow field using the available elements (see the examples)
     flow = 'radial_flow(x,[1 1],0.001,0.25) + circular_flow(x,[1 1],0,1) + 0.1*randn(1,2)';
     % flow = 'radial_flow(x,[1 1],0.001,0.25)+lateral_flow(x,1,2)';
     % flow = 'sinusoidal_flow(x,[0 0.5],[1 1],0);'
     
   % Take the objects, generate the frame, compute speeds, update locations, and repeat
     [sequence,locations,speeds] = dots_sequence(initial_locations,sizes,colors,flow,Nx,Nt,fs,ft); 
 
   % Build the matlab movie sequence from the data
     M = build_achrom_movie(sequence,0,255,75,1);  
     
     